function [sys,E]=OscAnHarm(L,Ep0,C,m,varargin)
% OSCANHARM donne les nergies des tats confins d'un oscillateur anharmonique (en x^3)
%   >> L    = largeur (nm) du puits
%   >> Ep0  = Profondeur (eV) du puits
%   >> C    = Constante de la perturbation
%   >> m    = Masse (par rapport  l'lectron) de l'objet physique
%   >> varargin     = Arguments d'entre optionnels
%           > nx    = Nombre de point selon Ox (dfaut : 100)
%           > nE    = Nombre de valeurs de l'nergie (dfaut : 500)
%           > Emin  = Valeur minimale de l'nergie (dfaut : min(Ep))
%           > Emax  = Valeur maximale de l'nergie (dfaut : 0)
%
% Olivier Pujol : septembre 2013

%% Vrifications diverses

if (length(varargin)>4), error('Mauvais nombre d''arguments d''entre optionnels');end
if (nargin-length(varargin)~=4),error('Mauvais nombre d''arguments d''entre non optionnels');end

%% Vrification de la valeur de C

if (C==0),error('C=0 ! C''est un oscillateur harmonique');end

Clim=16*abs(Ep0)/(3*sqrt(3)*L^3);
if (abs(C)>Clim),error('La valeur de C est trop grande');end        

clear Clim

%% Dtermination des limites du puits

% Coefficients du polynme (ordre dcroissant des puissances)
polynome=[-C 4*Ep0/L^2 0 -Ep0];     % Trois racines relles si abs(C)<Clim
racine=roots(polynome);             % Racines du polynme

if (isreal(racine)==0)              % On contrle que les racines soient toutes relles
    error('Il y a des racines complexes (la valeur de C est trop grande)');
end

% Les limites spatiales du puits sont donnes par les deux racines les plus
% petites. La racine la plus grande, racine(1), est ici sans intrt.

        if (C>0)
            xA=racine(3);    % Limite infrieure
            xB=racine(2);    % Limite suprieure
        elseif (C<0)
            xA=racine(2);    % Limite infrieure
            xB=racine(3);    % Limite suprieure
        end

if (xA>xB)
    error('Domaine spatial incorrect : la borne minimale est suprieure  la borne maximale');
end

clear polynome racine

%% Arguments d'entre optionnels

switch length(varargin)
    case 0
        nx=100;                     % Valeur par dfaut du nombre de points selon Ox
        nE=500;                     % Valeur par dfaut du nombre de valeurs de l'nergie
    case 1
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=500;                     % Valeur par dfaut du nombre de valeurs de l'nergie
    case 2
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=varargin{2};             % Valeur par dfaut du nombre de valeurs de l'nergie
    case 4
        nx=varargin{1};             
        nE=varargin{2};             
        E=linspace(varargin{3},varargin{4},nE);     % Ensemble des valeurs de l'nergie
    otherwise
        ('Mauvais nombre d''arguments d''entre optionnels');
end

%% Energie potentielle

% Construction
x=linspace(xA,xB,nx);                                   % Domaine spatial
clear xA xB
Epc=4*Ep0*(x/L).^2-C*x.^3;        % Energie potentielle (perturbation anharmonique cubique)

% Discrtisation
[M,xt,Ep]=discretisation(x,Epc);    % Dicrtisation

% Valeurs de l'nergie
if (length(varargin)~=4)
    E=linspace(min(Ep),Ep0,nE);       % Ensemble des valeurs de l'nergie
end    

if (max(E)>Ep0)
    error('Certaines valeurs de E sont hors du domaine des tats confins');
end

%% Etats confins

[sys,E]=Epot(M,xt,Ep,m,E);          % Fabrication du systme
sysconf(sys,E);                     % Etats confins

return;